/* $Id: srcoff.c,v 1.3 1998/09/16 02:05:20 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test that starting and stopping the source does not cause errors.
   We once had a report of occasional errors when stopping the source
   using e1432_set_active. */

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_CHANS)

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: srcoff.c,v 1.3 1998/09/16 02:05:20 ericb Exp $";
static const char *progname;

static void
my_usleep(unsigned int usec)
{
    struct timeval timeout;

    timeout.tv_sec = usec / 1000000;
    timeout.tv_usec = usec % 1000000;
    (void) select(0, 0, 0, 0, &timeout);
}

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].source_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_SOURCE_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

static int
setup(E1432ID hw, int group, int src_mode)
{
    CHECK(e1432_set_active(hw, group, E1432_CHANNEL_ON));
    CHECK(e1432_set_amp_scale(hw, group, 1));
    switch (src_mode)
    {
    case E1432_SOURCE_MODE_SINE:
    case E1432_SOURCE_MODE_RAND:
    case E1432_SOURCE_MODE_RANDZ:
	CHECK(e1432_set_ramp_rate(hw, group, 0.1));
	break;
    case E1432_SOURCE_MODE_BSINE:
    case E1432_SOURCE_MODE_BRAND:
    case E1432_SOURCE_MODE_BRANDZ:
	CHECK(e1432_set_ramp_rate(hw, group, 0.01));
	break;
    }
    CHECK(e1432_set_range(hw, group, 1));
    CHECK(e1432_set_span(hw, group, 10000));
    CHECK(e1432_set_source_mode(hw, group, src_mode));

    return 0;
}

static int
run(E1432ID hw, int group)
{
    CHECK(e1432_init_measure(hw, group));
    return 0;
}

static int
stop(E1432ID hw, int group, int mode, int src_mode)
{
    switch (mode)
    {
    case 0:
	/* Deactivating a source turns it off */
	CHECK(e1432_set_active(hw, group, E1432_CHANNEL_OFF));
	/* Reactivate it for next measurement */
	CHECK(e1432_set_active(hw, group, E1432_CHANNEL_ON));
	break;
    case 1:
	/* Reset measure turns off sources */
	CHECK(e1432_reset_measure(hw, group));
	break;
    case 2:
	/* Changing source mode turns off the source */
	CHECK(e1432_set_source_mode(hw, group, E1432_SOURCE_MODE_ARB));
	CHECK(e1432_set_source_mode(hw, group, E1432_SOURCE_MODE_BARB));
	/* Restore source mode for next measurement */
	CHECK(e1432_set_source_mode(hw, group, src_mode));
	break;
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uV] [-l loops] [-L laddr]\n"
		   "\t    [-m src_mode] [-n nchan] [-N nmod]\n"
		   "\t-l: Number of src on/off loops, default 10\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-m: Set source mode to <src_mode>, default RAND\n"
		   "\t-n: Use <nchan> channels, default all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    long    loops;
    char   *p;
    int     i, c, nmod, src_mode, stop_mode, group, nchan, verbose;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    loops = 10;
    laddr[0] = 8;
    src_mode = E1432_SOURCE_MODE_RAND;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "l:L:m:n:N:uvV")) != -1)
	switch (c)
	{
	case 'l':
	    loops = strtol(optarg, &p, 0);
	    if (optarg == p || loops < 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of loops: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    p = optarg;
	    while (*p != '\0')
	    {
		*p = tolower(*p);
		p++;
	    }
	    if (strcmp(optarg, "rand") == 0)
		src_mode = E1432_SOURCE_MODE_RAND;
	    else if (strcmp(optarg, "brand") == 0)
		src_mode = E1432_SOURCE_MODE_BRAND;
	    else if (strcmp(optarg, "sine") == 0)
		src_mode = E1432_SOURCE_MODE_SINE;
	    else if (strcmp(optarg, "bsine") == 0)
		src_mode = E1432_SOURCE_MODE_BSINE;
	    else if (strcmp(optarg, "randz") == 0)
		src_mode = E1432_SOURCE_MODE_RANDZ;
	    else if (strcmp(optarg, "brandz") == 0)
		src_mode = E1432_SOURCE_MODE_BRANDZ;
	    else
	    {
		(void) fprintf(stderr,
			       "%s: invalid source mode: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, src_mode) < 0)
	return EXIT_FAILURE;
    for (stop_mode = 0; stop_mode < 3; stop_mode++)
	for (i = 0; i < loops; i++)
	{
	    if (verbose)
		(void) printf("Stop mode %d loop %4d\n", stop_mode, i);
	    if (run(hw, group) < 0)
		return EXIT_FAILURE;
	    my_usleep(300000);
	    if (stop(hw, group, stop_mode, src_mode) < 0)
		return EXIT_FAILURE;
	    my_usleep(300000);
	}

    return EXIT_SUCCESS;
}
